/*
 * TMS7xxx pin exersizer
 */

#define NAME   "TMS7xxx exerciser V"
#define VERSION 0.5

#include "T7pins.h" 

#define SERIALBUFSIZE         50
char serialBuffer[SERIALBUFSIZE];
byte setBufPointer = 0;

#define LED 13

uint16_t repeatRate = 512;
#define AL_DELAY 10L
#define E_DELAY 10L
#define RECORDSIZE 16
#define DATARECORDTYPE 0

#define DUMPPAGE 0x00FF
unsigned int lastEndAddress = 0;

unsigned int addressOffset = 0;
#define EN_PULSE  10 // in microseconds
#define AL_PULSE 1 // in microseconds

#define PORT_OUT 0xFF;
#define PORT_IN  0x00;

// core routines

void setup() {
  Serial.begin(9600);
  Serial.print(NAME);
  Serial.println(VERSION, 1);
  
  pinMode(LED, OUTPUT);
  
  pinMode(T7RES,   INPUT_PULLUP);
  pinMode(T7IRQ1,   INPUT_PULLUP);
  pinMode(T7IRQ3,   INPUT_PULLUP);

  digitalWrite(T7EN,  LOW);
  digitalWrite(T7AL, LOW);
  digitalWrite(T7RW, HIGH);
  pinMode(T7EN,     OUTPUT);
  pinMode(T7AL,    OUTPUT);
  pinMode(T7RW,    OUTPUT);
 
  DDRA  = PORT_IN;  // address LSB /data input
  DDRC  = PORT_OUT; // address MSB output
  PORTK = PORT_OUT; // control out bits high
  DDRK  = PORT_OUT; // control out bits output
  DDRL  = PORT_IN;  // P1x bus input
  PORTL = PORT_OUT; // P1x bus pull ups
  
//  onlineReadMode();
  delay(1000);  
}  

void loop() {
  byte refreshAddress = 0;
  commandCollector();
}  


void clearSerialBuffer() {
  byte i;
  for (i = 0; i < SERIALBUFSIZE; i++) {
    serialBuffer[i] = 0;
  }
}

void commandCollector() {
  if (Serial.available() > 0) {
    int inByte = Serial.read();
    switch(inByte) {
    case '.':
//    case '\r':
    case '\n':
      commandInterpreter();
      clearSerialBuffer();
      setBufPointer = 0;
      break;
    case '\r':
      break;  // ignore carriage return
    default:
      serialBuffer[setBufPointer] = inByte;
      setBufPointer++;
      if (setBufPointer >= SERIALBUFSIZE) {
        Serial.println("Serial buffer overflow. Cleanup.");
        clearSerialBuffer();
        setBufPointer = 0;
      }
    }
  }
}

void commandInterpreter() {
    byte bufByte = serialBuffer[0];
    
    switch(bufByte) {
      case 'A':
      case 'a':
        testAddress();
        break;
      case 'B':
      case 'b':
        blinkPin();
        break;
      case 'D':  // dump memory
      case 'd':
        dumpMemory();
        break;
      case 'H':  // help
      case 'h':
      case '?':  // help
  //      Serial.println("F?:");
        usage();
        break; 
      case 'C':
      case 'c':
        calcChecksum(); // calculate various checksums
        break;
      case 'I':
      case 'i':
        generateDataRecords();
        generateEndRecord();
        break;
      case 'M':  // memory address read/write
      case 'm':
        readWriteMemory();
        break; 
      case 'P':
      case 'p':
        viewPorts();
        break;
      case 'Q':  // repeatRate
      case 'q':
        setRepeatRate();
        break; 
  //    case 'S':
  //    case 's':
  //      setValue(); // fill memory range with a value
  //      break;
      case 'T':
      case 't':
        testRAM();
        break;
      case 'W':
      case 'w':
        writePin();
        break;
      default:
        Serial.print(bufByte);
        Serial.print(" ");
        Serial.println("unsupported");
        return;
    }
}

void usage() {
    Serial.print("-- ");
    Serial.print(NAME);
    Serial.print(VERSION, 1);
    Serial.println(F(" command set --"));
    Serial.println(F("Aaaaa          - Set address bus to value aaaa"));
    Serial.println(F("Bpp or B#ss    - Blink pin p (in hex) or symbol: D0-D7,A8-AF,RW,EN,AL"));
    Serial.println(F("Cssss-eeee     - Generate checksums for address range"));
    Serial.println(F("D[ssss[-eeee]|+] - Dump memory from ssss to eeee (default 256 bytes)"));
    Serial.println(F("H              - This help text"));
    Serial.println(F("Issss-eeee     - Generate hex intel data records"));
    Serial.println(F("MRaaaa[+]      - Read memory address aaaa, optionally repeating"));
    Serial.println(F("MWaaaa vv[+]   - Write vv to address aaaa, optionally repeating"));
    Serial.println(F("Pp[+]          - View port/pins; A, C, L, c(omm), B/D/G"));
    Serial.println(F("Qn             - Blink/Repeat rate; 1, 2, 4, ..., 32678 ms (n=0-9,A-F)"));
    Serial.println(F("S[+]           - Signal view; RES, INT1, INT3"));
    Serial.println(F("Tssss-eeee     - Test RAM range (walking 1s)"));
    Serial.println(F("V              - View data bus, pins INT1, INT3, RESET"));
    Serial.println(F("Wpp v or W#ss v - Write pin (in hex) or symbol: AD0-AD7,A8-AF,RW,EN,AL; values 0, 1"));
    Serial.println(F("?              - This help text")); 
}

void dumpMemory() {
    unsigned long startAddress;
    unsigned long endAddress;
    bool repeatMode = 0;
    if (setBufPointer == 1 ) { // Automatic mode, dumping next page
        startAddress   = lastEndAddress == 0 ? 0 : lastEndAddress + 1;
        endAddress     = startAddress + DUMPPAGE;
        lastEndAddress = endAddress;
    } else if (setBufPointer == 2 && serialBuffer[1] == '+') { // continiously reading one page
        startAddress   = lastEndAddress - DUMPPAGE;
        endAddress     = startAddress + DUMPPAGE;
        lastEndAddress = endAddress;
        repeatMode = 1;
    } else if (setBufPointer == 5) { // dumping specified page
        startAddress   = get16BitValue(1);
        endAddress     = startAddress + DUMPPAGE;
        lastEndAddress = endAddress;
    } else if (setBufPointer == 10) { // dumping specified memory range
        startAddress   = get16BitValue(1);
        endAddress     = get16BitValue(6);
        lastEndAddress = endAddress;
    } else {
        Serial.println("unsupported"); 
    }
    unsigned char asChars[17];
    unsigned char *asCharsP = &asChars[0];
    unsigned char positionOnLine;
    asChars[16] = 0;
    do {
  //    printWord(startAddress);
  //    Serial.print("-");
  //    printWord(endAddress);
  //    Serial.println();
        unsigned long i;
        unsigned int data;
  
        for (i = startAddress; i <= endAddress; i++) {
            positionOnLine = i & 0x0F;
            if (positionOnLine == 0) {
                printWord(i);   // Address at start of line
                Serial.print(": ");
            }
            data = readByte(i);
            printByte(data);   // actual value in hex
            // fill an array with the ASCII part of the line
            asChars[positionOnLine] = (data >= ' ' && data <= '~') ? data : '.';
            if ((i & 0x03) == 0x03) Serial.print(" ");
            if ((i & 0x0F) == 0x0F) {
                Serial.print (" ");
                printString(asCharsP); // print the ASCII part
                Serial.println("");
            }
        }
        Serial.println();
        delay(repeatRate);
        if (stopIt()) repeatMode = 0;
    } while (repeatMode);
}

unsigned int readByte(unsigned int address) { // for muxed bus
    unsigned int data = 0;
// init
    // ALATCH      LOW
    digitalWrite(T7AL, LOW);
    // High Addr   undef
    // Low Addr    undef
    ADDR_MSB_DIR = PORT_OUT;
    // Enable      HIGH
    digitalWrite(T7EN, HIGH);
    // R/W*        LOW
    digitalWrite(T7RW, HIGH);

// read cycle
    setAddress(address);  // AL HIGH, set address, AL LOW
    // delay
    delay(AL_DELAY);
    // ENABLE      LOW
    digitalWrite(T7EN, LOW);
    ADDRDATA_DIR = PORT_IN;
    delay(E_DELAY);
    // Low Addr    D0-D7 (from external Memory)
    data = ADDRDATA_IN;
    // ENABLE      HIGH
    digitalWrite(T7EN, HIGH);

    return data; 
}

void writeByte(unsigned int address, unsigned int value) { // muxed bus
//Init:
    // ALATCH      LOW
    digitalWrite(T7AL, LOW);
    // High Addr   undef
    // Low Addr    undef
    ADDR_MSB_DIR = PORT_OUT;
    // Enable      HIGH
    digitalWrite(T7EN, HIGH);
    // R/W*        LOW
    digitalWrite(T7RW, LOW);

// Write cycle
    setAddress(address);  // AL HIGH, set address, delay, AL LOW
    ADDRDATA_DIR = PORT_OUT;
    // ENABLE      LOW
    digitalWrite(T7EN, LOW);
    delay(E_DELAY);
    // Low Addr    D0-D7
    ADDRDATA_OUT = value;
    delay(E_DELAY);
    // ENABLE      HIGH 
    digitalWrite(T7EN, HIGH);
    ADDRDATA_DIR = PORT_IN;
}


void printByte(unsigned char data) {
    unsigned char dataMSN = data >> 4;
    unsigned char dataLSN = data & 0x0F;
    Serial.print(dataMSN, HEX);
    Serial.print(dataLSN, HEX);
}

void printWord(unsigned int data) {
    printByte(data >> 8);
    printByte(data & 0xFF);
}

void dataBusReadMode() {
   DDRL  = 0x00;  // read mode
}

void dataBusWriteMode() {
   DDRL  = 0xFF;  // write mode
}

void blinkPin() {
    int pin;
    Serial.print("B");
    if (serialBuffer[1] == '#') {
        Serial.print("#");
        pin = getPinBySymbol();    
    } else {
        pin  = get8BitValue(1);
        Serial.print(serialBuffer[1]);
        Serial.print(serialBuffer[2]);
        Serial.print(" ");
    }
    Serial.print("Pin: ");
    Serial.println(pin, DEC);  
    pinMode(pin, OUTPUT);
    while(1) {
        digitalWrite(pin, !digitalRead(pin));
        delay(repeatRate);
        if (stopIt()) return;
    }
}

void writePin() {
    unsigned char pin;
    bool value;
    Serial.print("Setting pin ");
    if (serialBuffer[1] == '#') {
        pin = getPinBySymbol();    
        value = (serialBuffer[5] == 0x30);
        Serial.print(serialBuffer[2]);
        Serial.print(serialBuffer[3]);
        Serial.print(" (");
        Serial.print(pin, DEC);
        Serial.print(") to ");
        Serial.println(serialBuffer[5]);
    } else {
        pin  = get8BitValue(1);
        value = (serialBuffer[4] == 0x30);
        Serial.print(serialBuffer[1]);
        Serial.print(serialBuffer[2]);
        Serial.print(" (");
        Serial.print(pin, DEC);
        Serial.print(") to ");
        Serial.println(serialBuffer[4]);    
    }
    digitalWrite(pin, value ? 0 : 1);
}

unsigned char getPinBySymbol() {
    // B[0-F][0-F], B#A[8-F], B#AD[0-7], B#RW, B#E, B#AS, B#P1[0-7], B#P2[0-4]
    if (serialBuffer[2] == 'D' && serialBuffer[3] <= '7') {
        Serial.print("AD");
        Serial.write(serialBuffer[3]);
        Serial.print(" ");
        return serialBuffer[4] - '0' + T7AD0;
    } else if (serialBuffer[2] == 'A' && serialBuffer[3] > '7' && serialBuffer[3] <= 'F') {
        int i = serialBuffer[3] - '8';
        i = T7A8 - i;
        Serial.print("A");
        Serial.write(serialBuffer[3]);
        Serial.print(" ");
        return (i <= 28) ? i + 7 : i;
    } else if (serialBuffer[2] == 'R' && serialBuffer[3] == 'W') {
        Serial.print("RW ");  
        return T7RW;
    } else if (serialBuffer[2] == 'A' && serialBuffer[3] == 'L') {
        Serial.print("AL ");  
        return T7AL;
    } else if (serialBuffer[2] == 'E') {
        Serial.print("E ");  
        return T7EN;
    } else {
        Serial.println(" unknown symbol");
    }
    return 0;
}

void printBin(unsigned char value) {
    Serial.print((value & 0b10000000) ? "1" : "0");
    Serial.print((value & 0b01000000) ? "1" : "0");
    Serial.print((value & 0b00100000) ? "1" : "0");
    Serial.print((value & 0b00010000) ? "1" : "0");
    Serial.print((value & 0b00001000) ? "1" : "0");
    Serial.print((value & 0b00000100) ? "1" : "0");
    Serial.print((value & 0b00000010) ? "1" : "0");
    Serial.print((value & 0b00000001) ? "1" : "0");  
}

void printString(unsigned char *asCharP) {
    unsigned char i = 0;
    while(asCharP[i] != 0) {
        Serial.write(asCharP[i]); 
        i++;
    }
}

void readWriteMemory() {
    // MRaaaa[+], MWaaaa vv[+]
    uint16_t address;
    bool repeatMode = 0;
    address  = get16BitValue(2);
    if (serialBuffer[1] == 'R' || serialBuffer[1] == 'r') {
        if (setBufPointer == 7 && serialBuffer[6] == '+') {
            repeatMode = 1;
        }
        dataBusReadMode();
        do {
            Serial.print("MR ");
            Serial.print(address, HEX);
            Serial.print(": ");
            Serial.println(readByte(address), HEX);
            if (stopIt()) {
                return;
            }
            delay(repeatRate);
            if (stopIt()) repeatMode = 0;
        } while (repeatMode);
      } else if (serialBuffer[1] == 'W' || serialBuffer[1] == 'w') {
        uint8_t data;
        data  = get8BitValue(7);
        if (setBufPointer == 10 && serialBuffer[9] == '+') {
            repeatMode = 1;
        }
        dataBusWriteMode();
        do {
            Serial.print("MW ");
            Serial.print(address, HEX);
            Serial.print(": ");
            Serial.println(data, HEX);
            writeByte(address, data);
            if (stopIt()) {
                return;
            }
            delay(repeatRate);  
            if (stopIt()) repeatMode = 0;
        } while (repeatMode); 
    } else {
      Serial.print("not supported");
      return; 
    }
}

bool stopIt() {
    if (Serial.available() > 0) {
        clearSerialBuffer();
        setBufPointer = 0;
        return true;
    }
    return false;
}

void setRepeatRate() {
    if (setBufPointer == 2) {
        byte value = serialBuffer[1] - '0';
        value = (value > 9) ? (value - 7) : value;
        repeatRate = 1 << value;
        Serial.print("Repeat rate set to ");
        Serial.print(repeatRate, DEC);
        Serial.println(" ms");
    } else {
        Serial.println("not supported");
    }
}

void setAddress(unsigned int address) { // Sets upper and lower address byte, and pulses the address latch
    unsigned int addressLSB = address & 0xFF;
    unsigned int addressMSB = address >> 8;
    digitalWrite(T7AL, HIGH);
    ADDRDATA_DIR = PORT_OUT;
    ADDRDATA_OUT = addressLSB;
    ADDR_MSB = addressMSB;
    delay(AL_DELAY);
    digitalWrite(T7AL, LOW);
}

void testAddress() { // Test routine MSB address pins and LSB address latch
    // Aaaaa
    uint16_t address;
    bool repeatMode = 0;
    address  = get16BitValue(1);
    setAddress(address);
    Serial.print("A:");
    Serial.println(address,HEX);
}

void generateDataRecords() {
    unsigned int startAddress;
    unsigned int endAddress;
    startAddress  = get16BitValue(1);
    endAddress  = get16BitValue(6);
    printWord(startAddress);
    Serial.print("-");
    printWord(endAddress);
    Serial.println();
  
    unsigned int i, j;
    unsigned char addressMSB, addressLSB, data;
    unsigned char sumCheckCount = 0;
  
    dataBusReadMode();  
    for (i = startAddress; i < endAddress; i += RECORDSIZE) {
        sumCheckCount = 0;
        Serial.print(":");
        printByte(RECORDSIZE);  
        sumCheckCount -= RECORDSIZE;
        addressMSB = i >> 8;
        addressLSB = i & 0xFF;
        printByte(addressMSB);
        printByte(addressLSB);
        sumCheckCount -= addressMSB;
        sumCheckCount -= addressLSB;
        printByte(DATARECORDTYPE);
        sumCheckCount -= DATARECORDTYPE;
        for (j = 0; j < RECORDSIZE; j++) {
            data = readByte(i + j);
            printByte(data);
            sumCheckCount -= data;
      }
      printByte(sumCheckCount);
      Serial.println();
    }
}

void generateEndRecord() {
  Serial.println(":00000001FF");
}

unsigned int get16BitValue(byte index) {
    byte i = index;
    unsigned address;
    address  = getNibble(serialBuffer[i++]) * (1 << 12);
    address += getNibble(serialBuffer[i++]) * (1 << 8);
    address += getNibble(serialBuffer[i++]) * (1 << 4);
    address += getNibble(serialBuffer[i++]);
    return address;
}

byte get8BitValue(byte index) {
    byte i = index;
    byte data;
    data  = getNibble(serialBuffer[i++]) * (1 << 4);
    data += getNibble(serialBuffer[i++]);
    return data;
}

int getNibble(unsigned char myChar) {
    int nibble = myChar;
    if (nibble > 'F') nibble -= 0x20;  // lower to upper case
    nibble -= '0';
    if (nibble > 9) nibble -= 7; // offset 9+1 - A
    return nibble;
}

void viewPorts() { // A, B, C, L, c
    bool repeat = 0;
    if (serialBuffer[1] == '+') repeat = 1;
    byte oldValA = 0;
    byte valueA;
    byte oldValB = 0;
    byte valueB;
    byte oldValC = 0;
    byte valueC;
    byte oldValL = 0;
    byte valueL;
    byte oldValc = 0;

    PORTA = 0xFF; // pull up
    DDRA = PORT_IN;
    PORTB = 0xFF; // pull up
    DDRB = PORT_IN;
    PORTC = 0xFF; // pull up
    DDRC = PORT_IN;
    PORTL = 0xFF; // pull up
    DDRL = PORT_IN;

    do {
        valueA = PINA;
        if (oldValA != valueA) {
            Serial.print("A");
            if (repeat) Serial.print("+");
            Serial.print(": ");
            printBin(valueA);
            Serial.println();
            oldValA = valueA;
        }
        valueB  = digitalRead(T7EN);
        valueB += digitalRead(T7IRQ3);
        valueB += digitalRead(T7IRQ1);
        valueB += digitalRead(T7RW);
        valueB += digitalRead(T7AL);
        valueB += digitalRead(T7RES);
        if (oldValB != valueB) {
            Serial.print("B");
            if (repeat) Serial.print("+");
            Serial.print(": ");
            printBin(valueB);
            Serial.println();
            oldValB = valueB;
        }
        valueC = PINC;
        if (oldValC != valueC) {
            Serial.print("C");
            if (repeat) Serial.print("+");
            Serial.print(": ");
            printBin(valueC);
            Serial.println();
            oldValC = valueC;
        }
        valueL = PINL;
        if (oldValL != valueL) {
            Serial.print("L");
            if (repeat) Serial.print("+");
            Serial.print(": ");
            printBin(valueL);
            Serial.println();
            oldValL = valueL;
        }
        if (stopIt()) repeat = 0;
    } while(repeat);
}

byte memTestDataBus(uint16_t address)
{
    byte pattern;


    /*
     * Perform a walking 1's test at the given address.
     */
    for (pattern = 1; pattern != 0; pattern <<= 1)
    {
        /*
         * Write the test pattern.
         */
        writeByte(address, pattern);

        /*
         * Read it back (immediately is okay for this test).
         */
        if (readByte(address) != pattern) 
        {
            return (pattern);
        }
    }
    return (0);

}   /* memTestDataBus() */

byte testRAM() {
  unsigned int startAddress;
  unsigned int endAddress;
  unsigned char value;
  // Ussss eeee
  startAddress = get16BitValue(1);
  endAddress   = get16BitValue(6);
  Serial.print(F("Testing RAM range "));
  Serial.print(startAddress, HEX);
  Serial.print(F(" - "));
  Serial.println(endAddress, HEX);
  dataBusReadMode();
  unsigned int i;
  byte result;
  for (i = startAddress; i <= endAddress; i++) {
    result = memTestDataBus(i);
    if (result) {
      Serial.print(F("Failed with pattern "));
      printBin(result);
      Serial.print(F(" at "));
      Serial.print(i, HEX);
      Serial.print(F(" (read: "));
      printBin(readByte(i));
      Serial.println(")");
    }
    if (stopIt()) break;
  }
  dataBusReadMode();
  Serial.println("RAM test done.");
}

unsigned int blockChecksum(unsigned long startAddress, unsigned long endAddress)
{
  unsigned long checksum = 0;
  for (unsigned long i=startAddress; i<=endAddress; i++) {
    checksum += readByte(i);  
  }
  return checksum;
}

uint8_t andOrDiff(unsigned long startAddress, unsigned long endAddress)
{
  uint8_t checksum = 0;
  for (unsigned long i=startAddress; i<=endAddress; i++) {
    checksum = (readByte(i) & checksum) - (readByte(i) | checksum); //  P = (Q OR P) - (Q AND P)
  }
  return checksum;
}

void calcChecksum() {
  unsigned int startAddress;
  unsigned int endAddress;
  unsigned char value;
  // Cssss eeee
  startAddress = get16BitValue(1);
  endAddress   = get16BitValue(6);

  Serial.print(F("Checksum block "));
  Serial.print(startAddress, HEX);
  Serial.print(F("h - "));
  Serial.print(endAddress, HEX);
  Serial.print(F("h : "));
  unsigned long checkSum = blockChecksum(startAddress, endAddress);
  uint8_t andOrChecksum = andOrDiff(startAddress, endAddress);
  Serial.print(checkSum, HEX);
  Serial.print(F("h, "));
  Serial.print(checkSum, DEC);
  Serial.print(F(", "));
  Serial.print(checkSum & 0xFF, DEC);
  Serial.print(F(", and/or: "));
  Serial.print(andOrChecksum, DEC);  
  Serial.println();
}
