/***************************************************
  ADXL345 Digital Accelerometer driver for Arduino.  
   Based on the Adafruit ADXL345 breakout board sketch.
   http://www.adafruit.com/products/1231
  
  Datasheet:
   https://www.sparkfun.com/datasheets/Sensors/Accelerometer/ADXL345.pdf
   
  F.J. Kraan, fjkraan@xs4all.nl, 2013-12-30

 ***************************************************/

#include <Arduino.h>
#include "ADXL345.h"
#include "I2C.h"
#include "generic.h"

xyzInts_t       am;
xyzInts_t       *accelerometer = &am;
xyzBytes_xSB_t  aud;
xyzBytes_xSB_t  *aUnsortedData = &aud;

byte range;

xyzInts_t* readAccelXYZ() {
    read6Register(ADXL345_ADDRESS, ADXL345_REG_DATAX0, aUnsortedData);
    // ADXL345 has big endian register order 
    accelerometer->x = aUnsortedData->x1 << 8 | aUnsortedData->x0;
    accelerometer->y = aUnsortedData->y1 << 8 | aUnsortedData->y0;
    accelerometer->z = aUnsortedData->z1 << 8 | aUnsortedData->z0;

    return accelerometer;
}

byte ADXL345_getDeviceID(void) {
    // Check device ID register
    return readRegister(ADXL345_ADDRESS, ADXL345_REG_DEVID);
}

bool ADXL345_init(byte range) {
    /* Check connection */
    byte deviceid = ADXL345_getDeviceID();
    if (deviceid != ADXL345_ID) {
        /* No ADXL345 detected ... return false */
        return false;
    }
  
    // Enable measurements
    writeRegister(ADXL345_ADDRESS, ADXL345_REG_POWER_CTL, 0x08);  
  
    ADXL345_setRange(range);
    
    return true;
}

void ADXL345_setRange(byte newRange) {
    /* Read the data format register to preserve bits */
    uint8_t format = readRegister(ADXL345_ADDRESS, ADXL345_REG_DATA_FORMAT);

    /* Update the data rate */
    format &= ~0x0F;
    format |= newRange;
  
    /* Make sure that the FULL-RES bit is enabled for range scaling */
    format |= 0x08;
  
    /* Write the register back to the IC */
    writeRegister(ADXL345_ADDRESS, ADXL345_REG_DATA_FORMAT, format);
  
    /* Keep track of the current range (to avoid readbacks) */
    range = newRange;
}

byte ADXL345_getRange(void) {
    /* Read the data format register to preserve bits */
    return (readRegister(ADXL345_ADDRESS, ADXL345_REG_DATA_FORMAT) & 0x03);
}

void ADXL345_setDataRate(dataRate_t dataRate) {
    /* Note: The LOW_POWER bits are currently ignored and we always keep
       the device in 'normal' mode */
    writeRegister(ADXL345_ADDRESS, ADXL345_REG_BW_RATE, dataRate);
}

byte ADXL345_getDataRate(void) {
    return (readRegister(ADXL345_ADDRESS, ADXL345_REG_BW_RATE) & 0x0F);
}


