/* Name:    OT_listener
   Purpose: listen to OpenTherm (TM) communications between heater and thermostat and display
            results on a 4x16 LCD
   Author:  Martijn van den Burg, martijn@[remove-me-first]palebluedot . nl
   
   Copyright (C) 2009 Martijn van den Burg. All right reserved.

   This program is free software: you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/ 


/*
 * Made the listener work with the Arduino UNO and software version 1.0.5
 * Extended the Serial debug output
 * Changed display to 20x4, added more data
 *
 * Fred Jan Kraan, 2014-03-07, http://fjkraan.home.xs4all.nl/digaud/arduino/OpThMaster/
 */

#include <FiniteStateMachine.h>  // FSM library from http://www.arduino.cc/playground/Code/FiniteStateMachine
#include <LiquidCrystal.h>
#include <DS1307RTC.h>
#include <Time.h>
#include <Wire.h>
#include "OpTh.h"
#include "OpenTherm.h"

#define DEBUG 1    // enable Serial and LCD _debug_ output
#define LCDRAW 0   // 'raw' data output to LCD
#define LISTENER_VERSION 0.3

#define DEG (char)223  // degree character
#define BS  (char)0    // backslash


// The LCD has four rows of 20 characters
const int numCols = 20;
const int numRows = 4;

byte error_reading_frame;    // function return value

// Initialize the library with the numbers of the interface pins
LiquidCrystal lcd(12, 11, 7, 6, 5, 4);
//                RS  EN  4  5  6  7

//initialize FSM states
State GetPeriod      = State(get_period);
State DrawDisplay    = State(draw_display);
State WaitFrame      = State(wait_frame);
State GetFrame       = State(read_frame);
State DisplayFrame   = State(display_frame);


FSM OT_StateMachine = FSM(GetPeriod);     //initialize state machine, start in state: GetPeriod


OpTh OT = OpTh();  // create new OpTh class instance

byte backslash[8] = {
  0b00000,
  0b10000,
  0b01000,
  0b00100,
  0b00010,
  0b00001,
  0b00000,
  0b00000
};

void setup() {
  if (DEBUG) {
    Serial.begin(19200);
  }

  lcd.begin(numCols, numRows);
  lcd.setCursor(0, 0);
  lcd.print(" OpTh listener ");
  lcd.print(" 0.3");
  lcd.setCursor(3, 1);
  lcd.print("MvdBurg 2009");
  lcd.setCursor(1, 2);
  lcd.print("Ferroli adaptation");
  lcd.setCursor(3, 3);
  lcd.print("F.J.Kraan 2014");
  delay(1000);
  Serial.print("#OpTh listener start ");
  Serial.print(LISTENER_VERSION);
  Serial.println();
  
  pinMode(A1, INPUT);
  digitalWrite(A1, HIGH);

  lcd.createChar(0, backslash);

  OT.init();
}

void loop() {
  OT_StateMachine.update();         // trigger the initialization state.
  OT_StateMachine.transitionTo(GetPeriod).update();
  OT_StateMachine.transitionTo(DrawDisplay).update();

  while (1) {
    OT_StateMachine.transitionTo(WaitFrame).update();
    OT_StateMachine.transitionTo(GetFrame).update();
    OT_StateMachine.transitionTo(DisplayFrame).update();
  }
  
}





/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 *
 *    subroutines
 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* Get the period of the bits in the data stream. */
void get_period() {
  OT.measureOtPeriod();

  if (DEBUG) {
    long p = OT.getPeriod();
    
    Serial.print("Period: ");
    Serial.println((int)p);

    lcd.setCursor(0, 2);
    lcd.print("                    ");
    lcd.setCursor(0, 2);
    lcd.print("OT per.: ");
    lcd.print(p);
    lcd.print(" us");
    
    delay(2000);  // give user time to read the display
    lcd.clear();
  }
}



/* Put information on LCD. */
void draw_display() {
    lcd.clear();
    lcd.setCursor(0, 0);
    lcd.print("Room --.-/--.-");
    lcd.print(DEG);  
    lcd.print("C");

    lcd.setCursor(0, 1);
    lcd.print("Boiler --");
    lcd.print(DEG);  
    lcd.print("C");

    lcd.setCursor(0, 2); 
    lcd.print("CH  --");
    lcd.print(DEG);  
    lcd.print("C");
    
    lcd.setCursor(14, 1);
    lcd.print("Rl");

    lcd.setCursor(13, 2);
    lcd.print("Max");
    
    lcd.setCursor(13, 3);
    lcd.print("Min");
}


/* Wait for the start of a frame. The wait period, a period in which the signal is LOW. 
 * is determined by the kind of frame last received. 
 * The frame we encounter can either be a MASTER or a SLAVE frame.
 */
void wait_frame() {
  OT.waitFrame();
}  // wait_frame



/* Read the bits in the frame. 
 * Returns 0 for success, 1 for failure.
 */
void read_frame() {
  error_reading_frame = OT.readFrame();
}  // read_frame



/* Display information about heating settings and heater status on the LCD. 
 * Output is sent to Serial when DEBUG is 1.
 * Unprocessed/unformatted output is sent to the LCD when LCDRAW is 1.
 * Processed and formatted output is sent to the LCD when LCDRAW is 0.
 */
void display_frame() {
  bool lcDebug = !digitalRead(A1);
  byte msg_type = OT.getMsgType();
  byte data_id = OT.getDataId();
  unsigned int data_value = OT.getDataValue();
  static bool masterZeroMsgLast = 0;
  
  if (error_reading_frame && LCDRAW) {
    char *msg = OT.errmsg();
    lcd.clear();
    lcd.setCursor(1, 0);
    lcd.print(msg);
  }
  else {
    if (DEBUG) {
      printTimeStamp();
      Serial.print(" ");
      if (OT.isMaster() ) {
        Serial.print("M");
      }
      else {
        Serial.print("S");
      }
      Serial.print(messageType(msg_type));
      Serial.print(":");
      Serial.print((int)data_id);
      Serial.print(":");
//      Serial.println(data_value);
      formatPrintData(data_id, data_value, msg_type);
      Serial.println();
    }  // DEBUG
    // Detect missing slave message 0 and erase 
    if (masterZeroMsgLast && data_id != 0) {
      masterZeroMsgLast = 0;
      lcd.setCursor(18, 0); 
      lcd.print("  ");    // clear "Cf"
      lcd.setCursor(0, 3);
      lcd.print("        "); // clear "CH  FLAME" or "DHW FLAME"
    }
    if (lcDebug) {
      if (OT.isMaster() ) {
//        lcd.clear();
        lcd.setCursor(10, 2);
        lcd.print("M>        ");
        lcd.setCursor(13, 2);
        lcd.print(data_id, DEC);
        lcd.print(" ");
        lcd.print(data_value, DEC);
      } else {
        lcd.setCursor(10, 3);
        lcd.print("<S        ");
        lcd.setCursor(13, 3);
        lcd.print(data_id, DEC);
        lcd.print(" ");
        lcd.print(data_value, DEC);
      }  // master
    }  // LCDRAW

      if (OT.isMaster() ) rotator(17,0);
      switch(data_id) {
        case 1:  // Control setpoint
          if (OT.isMaster()) {
            lcdPrintFloat(4, 2, data_value);
            lcd.setCursor(8, 2);
            lcd.print(DEG);  
            lcd.print("C");
          }
        break;
        case 14:  // Maximum relative modulation level setting
          if (!OT.isMaster() && !lcDebug) {
            lcdPrintInt(16, 2, data_value);
            lcd.setCursor(19, 2);
            lcd.print("%");
          }
        break;
        case 15:  // Minimum boiler modulation level
          if (!OT.isMaster() && !lcDebug) {
            lcdPrintInt(16, 3, data_value);
            lcd.setCursor(19, 3);
            lcd.print("%");
          } 
        break;
        case 16:  // room setpoint
          if (OT.isMaster()) {
            lcdPrintFloat(5, 0, data_value);
            lcd.setCursor(9, 0);
            lcd.print("/");
          }
        break;
        case 17:  // Relative modulation level
          if (!OT.isMaster()) {
            lcdPrintInt(16, 1, data_value);
            lcd.setCursor(19, 1);
            lcd.print("%");
          }
        break;
        case 24:  // room actual temperature
          if (OT.isMaster()) {
            lcdPrintFloat(10, 0, data_value);
            lcd.setCursor(14, 0);
            lcd.print(DEG);
            lcd.print("C");
          }
        break;
        case 25:  // boiler temp
          if (! OT.isMaster()) {
            lcdPrintFloat(7, 1, data_value);
            lcd.setCursor(11, 1);
            lcd.print(DEG);
            lcd.print("C");
          }
        break;
        case 0:  // status
          if (! OT.isMaster()) {
            lcd.setCursor(0, 3);
            if (data_value & S_DHW_ACTIVE) {  // Domestic Hot Water on
              lcd.print("DHW");
            } else if (data_value & S_CH_ACTIVE) { // Central Heating on
              lcd.print("CH ");
            } else {             // only reached when the zero slave message flags is non-zero
              lcd.print("   ");
            }            
            lcd.setCursor(18, 0);
            if (data_value & S_COMF_ACTIVE) { // Comfort flag
              lcd.print("Cf");
            } else {             // only reached when the zero slave message flags is non-zero
              lcd.print("  ");
            }
            lcd.setCursor(4, 3);
            if (data_value & S_FLAME) { // 
              lcd.print("FLAME");
            } else {             // only reached when the zero slave message flags is non-zero
              lcd.print("     ");
            }
            lcd.setCursor(0, 3);
            if (data_value & S_FAULT) { // 
              lcd.print("FAULT     ");
            }
            masterZeroMsgLast = 0;
          } else {
            masterZeroMsgLast = 1;
          }
          break;
      }  // switch
    
 //   }
    
  }  // no frame error
  
}

// Assuming only values below 100, reasonable for temperatures. 
// No provision for negative numbers
void lcdPrintFloat(uint8_t column, uint8_t line, uint16_t data_value) {
    float t = (float)data_value / 256;
    lcd.setCursor(column, line);
    if (t < 10) { // just one digit
        lcd.print(" "); 
    }
    lcd.print(t);
    if (data_value % 256 == 0) {
        lcd.setCursor(column + 2, line);
        lcd.print(".0");
    }
}  

// Assuming integer percentages
void lcdPrintInt(uint8_t column, uint8_t line, uint16_t data_value) {
    uint8_t t = data_value / 256;
    lcd.setCursor(column, line);
    if (t < 100 && t >= 10) { // two digits
        lcd.print(" ");
    } else if (t <= 10) { // just one digit
        lcd.print("  "); 
    } 
    lcd.print(t);
}

// 4.2.2 Message Type - MSG-TYPE
char* messageType(byte msg_type) {
    if (msg_type == READ_DATA)    return "r";  // M: read data
    if (msg_type == WRITE_DATA)   return "w";  // M: write data
    if (msg_type == INVALID_DATA) return "?";  // M: invalid data
    if (msg_type == RESERVED)     return "-";  // M: reserved
    if (msg_type == READ_ACK)       return "r";  // S: read acknowledge
    if (msg_type == WRITE_ACK)      return "w";  // S: write acknowledge
    if (msg_type == UNKNOWN_DATAID) return "?";  // S: data invalid
   return "-";                                   // S: unknown dataID
}

int8_t getIntPart(uint16_t value) {
//    return value / 256;
    return value >> 8;
}

uint8_t getFractionalPart(uint16_t value) {
//    return value - (getIntPart(value) * 256);
    uint8_t frac = (value & 0xFF);
    return (frac >= 128) ? 5 : 0;
} 

void formatPrintData(uint8_t data_id, uint16_t data_value, uint8_t msg_type) {
    switch (data_id) {
    case STATUS_FLAGS:    // 0  R
      Serial.print(data_value);
      Serial.print(" ");
      formatPrintId0(data_value, msg_type);
      break;
    case MAX_REL_MOD:     // 14  W
    case MAX_CAP_MIN_LVL: // 15 R
    case REL_MOD_LVL:     // 17 R
       Serial.print(getIntPart(data_value));
      Serial.print(".");
      Serial.print(getFractionalPart(data_value));
      Serial.print(" %");
      break;   
    case CTRL_SP:         // 1   W
    case ROOM_SP:         // 16  W
    case ROOM_ATEMP:      // 24  W
    case BOILER_WTEMP:    // 25 R
      Serial.print(getIntPart(data_value));
      Serial.print(".");
      Serial.print(getFractionalPart(data_value));
      Serial.print(" .C");
      break;
    default:              // 9, 27, 29, 57
      Serial.print(data_value, DEC);
    }
}

void formatPrintId0(uint16_t data_value, uint8_t msg_type) {
  if (msg_type == READ_DATA) {
    Serial.print("CH ");
    Serial.print((data_value & M_CH_ENABLE) ? "enable; " : "disable; ");

    Serial.print("DHW ");
    Serial.print((data_value & M_DHW_ENABLE) ? "enable; " : "disable; ");

    Serial.print("Comf ");
    Serial.print((data_value & M_COMF_ENABLE) ? "enable" : "disable");
  } else if (msg_type == READ_ACK) {
    Serial.print((data_value & 0x001) ? "Fault! " : "");

    Serial.print("CH ");
    Serial.print((data_value & S_CH_ACTIVE) ? "" : "not ");
    Serial.print("active; ");

    Serial.print("DHW ");
    Serial.print((data_value & S_DHW_ACTIVE) ? "" : "not ");
    Serial.print("active; ");

    Serial.print("Flame ");
    Serial.print((data_value & S_FLAME) ? "on;" : "off; ");

    Serial.print("Comf ");
    Serial.print((data_value & M_COMF_ENABLE) ? "" : "not ");    
    Serial.print("active");
  } else {
    Serial.print(data_value, HEX);
  }
}

void print2digits(int number) {
  if (number >= 0 && number < 10) {
    Serial.write('0');
  }
  Serial.print(number);
}

void rotator(uint8_t column, uint8_t line) {
    static char rotator = ' ';
 
    if (rotator == 0x0) { 
      rotator = '|';
    } else if (rotator == '|') {
      rotator = '/';
    } else if (rotator == '/') {
      rotator = '-';
    } else {
      rotator = 0x0; // a  backslash
    }
    lcd.setCursor(column, line);
    lcd.print(rotator);      
}

void printTimeStamp() {
  tmElements_t tm;
  RTC.read(tm);
  Serial.print(tmYearToCalendar(tm.Year)); 
  Serial.print("-");
  print2digits(tm.Month);
  Serial.print("-");
  print2digits(tm.Day);
  Serial.print(" ");
  print2digits(tm.Hour);
  Serial.print(":");
  print2digits(tm.Minute);
  Serial.print(":");
  print2digits(tm.Second);
}  


/* End */
