#include <WProgram.h>
#include <Wire.h>
#include <DS1307new.h>
#include <EEPROM.h>
#include "SLC.h"

unsigned long previousMillis = 0;
#define UPDATE_INTERVAL  5
unsigned int secondCounter = 0;
unsigned int updateInterval;

#define SERIALBUFSIZE         11
char serialBuffer[SERIALBUFSIZE];
byte setBufPointer = 0;
unsigned int LDRCurrentLevel;
unsigned int LDRMorningSwitchLevel;
unsigned int LDREveningSwitchLevel;

// instantiate structs
time_t time;
date_t date;

unsigned long currentTime;
unsigned long morningSwitchTime;
unsigned long eveningSwitchTime;

char* unsupported = "unsupported";
char* progName = "Light & Time Switch ";
char* progVersion = "V0.95";
char str[5];
char dateTimeStr[11];

byte overrule = 0;
byte lightLevelOn = 1;
byte timeZoneOn = 1;
byte buttonState = HIGH;
byte lastButtonState = HIGH;
unsigned long lastDebounceTime = 0;
unsigned long debounceDelay = 50;
byte SQWLevel;
byte lastSQWLevel = 1;

// core routines

void setup() {
  Serial.begin(38400);
  Serial.print(progName);
  
  if (!ds1307Present()) {
    Serial.println("\nNo DS1307 RTC found. Halting");
    ltsHalted();
  }
  
  RTC.stopClock();
  setSQW(0);  // set to 1 Hz
  RTC.startClock();
  updateInterval = getUpdateInterval(); 
  if (updateInterval > 1000) updateInterval = UPDATE_INTERVAL;
  pinMode(LAMP, OUTPUT);
  pinMode(BUTTON, INPUT);
  pinMode(LED_L, OUTPUT);
  pinMode(LED_T, OUTPUT);
  digitalWrite(BUTTON, HIGH);
  digitalWrite(LED_L, LOW);
  digitalWrite(LED_T, LOW);

  LDRMorningSwitchLevel = getMorningSwitchLevel();
  LDREveningSwitchLevel = getEveningSwitchLevel();
  morningSwitchTime = getMorningSwitchTime();
  eveningSwitchTime = getEveningSwitchTime();

  digitalWrite(LED_L, HIGH);
  digitalWrite(LED_T, HIGH);
  Serial.println(progVersion);
/*  Serial.print("freeMem/Ram:");
  Serial.print(freeMem());
  Serial.print("/");
  Serial.println(freeRam()); */
  Serial.print("Update interval: ");
  Serial.print(updateInterval);
  Serial.println(" s");
  
  delay(1000);  
}

void setSQW(int rate){  // set the square wave output on pin 7 of the DS1307 chip
  rate = rate + 144;                         // add 0x90 (dec 144) to rate for 1 Hz
  RTC.ctrl = rate;
  RTC.setCTRL();
} 

void loop() {
  if (secondCounter <= 0) {
    RTC.getTime();
    date.year   = RTC.year;
    date.month  = RTC.month;
    date.date   = RTC.day;
    time.hour   = RTC.hour;
    time.minute = RTC.minute;
    time.second = RTC.second;
    unsigned long lHour = time.hour; // the lHour * 3600 is temporary stored in a variable of the type of lHour
    currentTime = (lHour * 3600 + time.minute * 60 + time.second);
    
    businessLogic();
    previousMillis = millis();
    secondCounter = updateInterval;
  } 
  SQWLevel = digitalRead(A3);
  if (SQWLevel == 0 && lastSQWLevel == 1) {
    secondCounter--;
    secondCounter = secondCounter % updateInterval;
    digitalWrite(LED_BASE, !digitalRead(LED_BASE));
  }
  lastSQWLevel = SQWLevel;
  commandCollector();
}  
 
void businessLogic() {
    LDRCurrentLevel = getLDRCurrentLevel();
    previousMillis = millis();
    printDateToSerial();
    Serial.print(" ");
    printTimeToSerial();
    Serial.print(" ");
    Serial.print(currentTime);
    Serial.print(" ");
    Serial.print(morningSwitchTime);
    Serial.print(" ");
    Serial.print(eveningSwitchTime);
    Serial.print(" ");
    Serial.print(LDRCurrentLevel);
    Serial.print(" ");
    Serial.print(LDRMorningSwitchLevel);
    Serial.print(" ");
    Serial.print(LDREveningSwitchLevel);
    Serial.print(" ");
    
    // Time condition
    if (currentTime > morningSwitchTime && currentTime < eveningSwitchTime) {
      digitalWrite(LED_T, LOW);
      timeZoneOn = 1;
    } else {
      digitalWrite(LED_T, HIGH);
      timeZoneOn = 0;
    } 
    // Light condition
    if (LDRCurrentLevel > LDRMorningSwitchLevel) {
      digitalWrite(LED_L, LOW);
      lightLevelOn = 1;
    }
    if (LDRCurrentLevel < LDREveningSwitchLevel) {
      digitalWrite(LED_L, HIGH);
      lightLevelOn = 0;
    }
    // Switch on if both conditions are true
    if (timeZoneOn == 1 && lightLevelOn == 1) {
      digitalWrite(BACKLIT, LOW);
      digitalWrite(LAMP, LOW);
    }
    // Switch off if either conditions are false
    if (timeZoneOn == 0 || lightLevelOn == 0) {
      digitalWrite(BACKLIT, HIGH);
      digitalWrite(LAMP, HIGH);        
    }
    Serial.print(getUpdateInterval());
    Serial.print(" ");
    Serial.print(digitalRead(LAMP) ? "off" : "on");
    Serial.println();
}

// delay without delay
boolean passed(unsigned int interval) {
  return (millis() - previousMillis > interval);
}

// wrappers around the array happy RAM getter and setter
unsigned int rtcRAMread(unsigned int address) {
  uint8_t value[1];
  RTC.getRAM(address, value, 1);
  return (unsigned int)value[0];
}

void rtcRAMwrite(unsigned int address, unsigned int value) {
  uint8_t val[1];
  val[0] = (uint8_t)value;
  RTC.setRAM((uint8_t)address, val, 1);
}

// wrapper LDR values
inline unsigned int getEveningSwitchLevel() {
  return(rtcRAMread(LDR_EVENING_VALUE_ADDR) * 4);
}

inline void setEveningSwitchLevel(unsigned int level) {
  rtcRAMwrite(LDR_EVENING_VALUE_ADDR, level / 4);
}

inline unsigned int getMorningSwitchLevel() {
  return(rtcRAMread(LDR_MORNING_VALUE_ADDR) * 4);
}

inline void setMorningSwitchLevel(unsigned int level) {
  rtcRAMwrite(LDR_MORNING_VALUE_ADDR, level / 4); 
}

inline unsigned int getLDRCurrentLevel() {
  return (MAX_ANALOG - analogRead(LDR0));
}

inline unsigned int getUpdateInterval() {
  return (unsigned int)rtcRAMread(UPDATE_INTERVAL_ADDR) * 4;
}

inline void setUpdateInterval(unsigned int interval) {
  rtcRAMwrite(UPDATE_INTERVAL_ADDR, interval / 4);
}

inline unsigned long getMorningSwitchTime() {
  unsigned int msb = (byte)rtcRAMread(MORNING_SECS_ADDR_MSB);
  unsigned int lsb = (byte)rtcRAMread(MORNING_SECS_ADDR_LSB);
  return (msb * 256 + lsb) * 2;
}

inline void setMorningSwitchTime(unsigned long secs) {
  secs = secs / 2;
  byte msb = secs / 256;
  byte lsb = secs % 256;
  rtcRAMwrite(MORNING_SECS_ADDR_MSB, msb); 
  rtcRAMwrite(MORNING_SECS_ADDR_LSB, lsb);
}

inline unsigned long getEveningSwitchTime() {
  unsigned int msb = (byte)rtcRAMread(EVENING_SECS_ADDR_MSB);
  unsigned int lsb = (byte)rtcRAMread(EVENING_SECS_ADDR_LSB);
  return (msb * 256 + lsb) * 2;
}

inline void setEveningSwitchTime(unsigned long secs) {
  secs = secs / 2;
  byte msb = secs / 256;
  byte lsb = secs % 256;
  rtcRAMwrite(EVENING_SECS_ADDR_MSB, msb); 
  rtcRAMwrite(EVENING_SECS_ADDR_LSB, lsb);
}

// output formatting
inline void printDateToSerial() {
  sprintf(dateTimeStr, "%04d-%02d-%02d", date.year, date.month, date.date);
  Serial.print(dateTimeStr);
} 

inline void printTimeToSerial() {
  sprintf(dateTimeStr, "%02d:%02d:%02d", time.hour, time.minute, time.second);
  Serial.print(dateTimeStr);
}

inline void printTimeFromSecs(unsigned long timeInSecs) {
  unsigned int hour = timeInSecs / 3600;
  unsigned int minute = (timeInSecs / 60) - hour * 60;
  unsigned int second = timeInSecs - (hour * 3600) - (minute * 60);
  sprintf(dateTimeStr, "%02d:%02d:%02d", hour, minute, second);
  Serial.print(dateTimeStr);
}

// interactive command part
inline unsigned long hhmmss2sec(byte hours, byte minutes, byte seconds) {
  unsigned long lHours = hours;
  unsigned int lMinutes = minutes;
  unsigned long total = seconds + lMinutes * 60 + lHours * 3600;
  return total;
}

void commandCollector() {
  if (Serial.available() > 0) {
    int inByte = Serial.read();
    switch(inByte) {
    case '.':
    case '\r':
    case '\n':
      commandInterpreter();
      clearSerialBuffer();
      setBufPointer = 0;
      break;
    default:
      serialBuffer[setBufPointer] = inByte;
      setBufPointer++;
      if (setBufPointer >= SERIALBUFSIZE) {
        Serial.println("Serial buffer overflow. Cleanup.");
        clearSerialBuffer();
        setBufPointer = 0;
      }
    }
  }
}

void commandInterpreter() {
  byte bufByte = serialBuffer[0];
  switch(bufByte) {
    case 'D':  // get/set date
    case 'd':
      dArgInterpreter();
      break; 
    case 'E':  // get/set evening switch time
    case 'e':
      eArgInterpreter();
      break;
    case 'H':  // help
    case 'h':
    case '?':  // help
//      Serial.println("EDHLMTUV?");
      usage();
      break; 
    case 'I':  // get/set logging update interval
    case 'i':
      iArgInterpreter();
      break;
    case 'L':  // get/set evening switch light level
    case 'l':
      lArgInterpreter();
      break;
    case 'M':  // get/set morning switch time
    case 'm':
      mArgInterpreter();
      break;
    case 'T':  // get/set time
    case 't':
      tArgInterpreter();
      break;
    case 'U':  // get/set morning switch light level
    case 'u':
      uArgInterpreter();
      break;
    case 'V':  // print version
    case 'v':
      Serial.println(progVersion);
      break;
    default:
      Serial.print(bufByte);
      Serial.print(" ");
      Serial.println(unsupported);
      return;
  }
}

void dArgInterpreter() {
  if (setBufPointer == 1) {
    printDateToSerial();
    Serial.println();
  } else if (setBufPointer == 9) {
    RTC.getTime();
    RTC.stopClock();
    RTC.year  = (serialBuffer[1]-'0') * 1000 + (serialBuffer[2]-'0') * 100 +
                (serialBuffer[3]-'0') * 10 + serialBuffer[4]-'0';
    RTC.month = (serialBuffer[5]-'0') * 10 + serialBuffer[6]-'0';
    RTC.day   = (serialBuffer[7]-'0') * 10 + serialBuffer[8]-'0';
    RTC.setTime();
    RTC.startClock();
    printDateToSerial();
    Serial.println(); 
    secondCounter = 0;
  } else {
    Serial.println(unsupported);
  }
} 

void tArgInterpreter() {
  if (setBufPointer == 1) {
    printTimeToSerial();
    Serial.println();
  } else if (setBufPointer == 7) {
    RTC.getTime();
    RTC.stopClock();
    RTC.hour   = (serialBuffer[1]-'0') * 10 + serialBuffer[2]-'0';
    RTC.minute = (serialBuffer[3]-'0') * 10 + serialBuffer[4]-'0';
    RTC.second = (serialBuffer[5]-'0') * 10 + serialBuffer[6]-'0';
    RTC.setTime();
    RTC.startClock();
    printTimeToSerial();
    Serial.println();
    secondCounter = 0;
  } else {
    Serial.println(unsupported);
  }
}

void iArgInterpreter() {
  if (setBufPointer == 1) {
    Serial.print(getUpdateInterval());
    Serial.println();
  } else if (setBufPointer == 4) {
    updateInterval = (serialBuffer[1]-'0') * 100 + 
                            (serialBuffer[2]-'0') * 10 + 
                            serialBuffer[3]-'0';
    setUpdateInterval(updateInterval);
    Serial.print(getUpdateInterval());
    Serial.println();    
  } else {
    Serial.println(unsupported);
  }
}

void mArgInterpreter() {
  byte hour;
  byte minute;
  byte second;
  if (setBufPointer == 1) {
    printTimeFromSecs(morningSwitchTime);
    Serial.println();
  } else if (setBufPointer == 7) {
    hour   = (serialBuffer[1]-'0') * 10 + serialBuffer[2]-'0';
    minute = (serialBuffer[3]-'0') * 10 + serialBuffer[4]-'0';
    second = (serialBuffer[5]-'0') * 10 + serialBuffer[6]-'0';
    morningSwitchTime = hhmmss2sec(hour, minute, second);
    setMorningSwitchTime(morningSwitchTime);
    printTimeFromSecs(morningSwitchTime);
    Serial.print(" (");
    Serial.print(morningSwitchTime);
    Serial.println(")");
  } else {
    Serial.print(unsupported);
  }
}

void eArgInterpreter() {
  byte hour;
  byte minute;
  byte second;
  if (setBufPointer == 1) {
    printTimeFromSecs(eveningSwitchTime);
    Serial.println();
  } else if (setBufPointer == 7) {
    hour   = (serialBuffer[1]-'0') * 10 + serialBuffer[2]-'0';
    minute = (serialBuffer[3]-'0') * 10 + serialBuffer[4]-'0';
    second = (serialBuffer[5]-'0') * 10 + serialBuffer[6]-'0';  
    eveningSwitchTime = hhmmss2sec(hour, minute, second);
    setEveningSwitchTime(eveningSwitchTime);
    printTimeFromSecs(eveningSwitchTime);
    Serial.print(" (");
    Serial.print(eveningSwitchTime);
    Serial.println(")");
  } else {
    Serial.print(unsupported);
  }
} 

void uArgInterpreter() {
  if (setBufPointer == 1) {
    Serial.println(LDRMorningSwitchLevel);
  } else if (setBufPointer == 4) {
    LDRMorningSwitchLevel = (serialBuffer[1]-'0') * 100 + 
                            (serialBuffer[2]-'0') * 10 + 
                            serialBuffer[3]-'0';
    setMorningSwitchLevel(LDRMorningSwitchLevel);
    LDRMorningSwitchLevel = getMorningSwitchLevel(); // truncates LSBits 
    Serial.println(LDRMorningSwitchLevel);
  } else {
    Serial.print(unsupported);
  }
}

void lArgInterpreter() {
  if (setBufPointer == 1) {
    Serial.println(LDREveningSwitchLevel);
  } else if (setBufPointer == 4) {
    LDREveningSwitchLevel = (serialBuffer[1]-'0') * 100 + 
                            (serialBuffer[2]-'0') * 10 + 
                            serialBuffer[3]-'0';
    setEveningSwitchLevel(LDREveningSwitchLevel);
    LDREveningSwitchLevel = getEveningSwitchLevel(); // truncates LSBits 
    Serial.println(LDREveningSwitchLevel);
  } else {
    Serial.print(unsupported);
  }
}

void clearSerialBuffer() {
  byte i;
  for (i = 0; i < SERIALBUFSIZE; i++) {
    serialBuffer[i] = 0;
  }
}

//void transferEEP2RTCR() {
//  int i;
//  for (i=0; i < 10; i++) {
//    Serial.print("Before: ");
//    Serial.print(" (");
//    Serial.print(i);
//    Serial.print(") ");
//    Serial.print(rtcRAMread(i));
//    Serial.print("  ");
//    rtcRAMwrite(i, EEPROM.read(i));
//    Serial.print("  After: ");
//    Serial.print(rtcRAMread(i));
//    Serial.println();
//  }
//}

int ds1307Present() {
  Wire.beginTransmission(DS1307_CTRL_ID);
  Wire.send(0x00);                     
  if (Wire.endTransmission() == 0) return TRUE;
  return FALSE;
}

void ltsHalted() {
  while (1) {
  digitalWrite(LED_L, HIGH);
  digitalWrite(LED_T, HIGH);
  digitalWrite(LED_BASE, HIGH);
  delay(250);    
  digitalWrite(LED_L, LOW);
  digitalWrite(LED_T, LOW);
  digitalWrite(LED_BASE, LOW);
  delay(250);    
  }
}

void usage() {
  Serial.println("E[hhmmss]   - Evening switch time");
  Serial.println("D[yyyymmdd] - Date");
  Serial.println("H           - This help text");
  Serial.println("L[nnn]      - Evening swich light level");
  Serial.println("M[hhmmss]   - Morning switch time");
  Serial.println("T[hhmmss]   - Time");
  Serial.println("U[nnn]      - Morning switch light level");
  Serial.println("V           - Version number");
  Serial.println("?           - This help text"); 
}

